/* dcf.c - receive DCF77 signal and adjust Linux system clock
 * Copyright Harald Koenig <koenig@tat.physik.uni-tuebingen.de> 1994
 */

/* permission is hereby granted to copy, modify and redistribute this code
 * in terms of the GNU Library General Public License, Version 2 or later,
 * at your option.
 */


#define __USE_BSD
#define __USE_BSD_SIGNAL
#define COMPILE_ADJTIME
#define COMPILE_ADJTIMEX

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/time.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include <termio.h>
#include <sys/ioctl.h>
#include <getopt.h>

#ifdef CIOGETEV
#include <linux/ppsclock.h>
#else

#define CIOGETEV  0
#define CIOGETNEV 0

struct ppsclockev {
	struct timeval tv;
	u_int serial;
};

struct ppsclocknev {
	struct timeval tv_data;
	u_int serial_data;
	struct timeval tv_raise;
	u_int serial_raise;
	struct timeval tv_fall;
	u_int serial_fall;
};
#endif

#define _ASM_IO_H
/* #include <linux/mc146818rtc.h> */
#include "mc146818rtc.h"
#define RTC_ALWAYS_BCD 1
#define outb_p(v,p) outb(p,v)
#define inb_p(p)    inb(p) 

#include <linux/timex.h>

#ifdef COMPILE_ADJTIMEX
#include <linux/unistd.h>
inline static _syscall1(int, adjtimex, struct timex *, ntx);
#endif

#ifdef COMPILE_ADJTIME

int adjtime(struct timeval * itv, struct timeval * otv)
{
  struct timex tntx;
  int result;
     
  tntx.modes = 0;
  if (itv) {
    tntx.offset = itv->tv_usec;
    tntx.modes = ADJ_OFFSET;
  }
  result = adjtimex(&tntx);
  if (result > 0) result = 0;
				  
  if (otv) {
    otv->tv_usec = tntx.offset;
    otv->tv_sec = 0;
  }
  return result;
}
#endif  /* COMPILE_ADJTIME */



char *wtag_txt[] = { "So","Mo","Di","Mi","Do","Fr","Sa","So" };

int setUNIX=0, setCMOS=0, setUNIXonce=0, setCMOSonce=0, debug=0, stats=0;
int dst, dstchange, last_dst;
int set_rts=0, set_dtr=1;

/* -------- terminal line setup ------------------------------- */

#define DCF_TTY "/dev/dcf77"

#define JungHans

#ifdef JungHans
  int dcf_baud=B50;
#else
  int dcf_baud=EXTB;
#endif

int c2b=0xc0;

int  dcf_dev;

#if CIOGETNEV
  int use_dcf_discipline = 1;
#else
  int use_dcf_discipline = 0;
#endif

void dcf_init(char *dcf_line)
{
  int i;
  struct termio dcf_termio;
  struct termios io;

  if ((dcf_dev = open(dcf_line, O_RDWR | O_NOCTTY )) < 0) {
    perror("DCF-77 Ansteuerung: open "); exit(-1);
  }

  dcf_termio.c_iflag     =    0;
  dcf_termio.c_oflag     =    0;
  dcf_termio.c_cflag     = dcf_baud | CS8 | CREAD | CLOCAL ;
  dcf_termio.c_lflag     = NOFLSH;
  dcf_termio.c_line      =    0;
  for(i=0; i<NCC; i++) 
    dcf_termio.c_cc[i]   = 0;
  dcf_termio.c_cc[VMIN]  = 1;
  dcf_termio.c_cc[VTIME] = 0;

  if (ioctl(dcf_dev, TCSETA, &dcf_termio) < 0) {
    perror("DCF-77 Ansteuerung: ioctl "); exit(-1);
  }

  i = TIOCM_DTR; ioctl(dcf_dev,set_dtr?TIOCMBIS:TIOCMBIC,&i);  /*  set  DTR */
  i = TIOCM_RTS; ioctl(dcf_dev,set_rts?TIOCMBIS:TIOCMBIC,&i);  /* clear RTS */

  tcgetattr(dcf_dev,&io);
  io.c_lflag &= ~ICANON;
  tcsetattr(dcf_dev,TCSANOW,&io);
}



/* ----------- clock stuff ------------------------------------ */

#define DCF_DIFF(t1,t2) ( (t1.tv_sec -t2.tv_sec)*1000000+(t1.tv_usec-t2.tv_usec) )
#define PF(t) t.tv_sec, t.tv_usec

/* ----- CMOS clock I/O routines ------------------------------ */

#define ASM_IO

#ifndef ASM_IO
int cmos_fd;
#endif

static inline void outb (short port, char val)
{
#ifdef ASM_IO
  __asm__ volatile ("out%B0 %0,%1" : :"a" (val), "d" (port));
#else
  lseek(cmos_fd, port, 0);
  write(cmos_fd, &val, 1);
#endif
}

static inline unsigned char inb (short port)
{
  unsigned char ret;
#ifdef ASM_IO
  __asm__ volatile ("in%B0 %1,%0" : "=a" (ret) : "d" (port));
#else
  lseek(cmos_fd, port, 0);
  read (cmos_fd, &ret, 1);
#endif
  return ret;
}

void cmos_init()
{
#ifdef ASM_IO
  ioperm(0x70, 2, 1);
#else
  cmos_fd = open("/dev/port", 2);
  if (cmos_fd < 0) {
    perror("unable to open /dev/port read/write : ");
    exit(1);
  }
  if (lseek(cmos_fd, 0x70, 0)<0 || lseek(cmos_fd, 0x71, 0)<0) {
    perror("unable to seek port 0x70 in /dev/port : ");
    exit(1);
  }
#endif
}

static inline int cmos_read(int addr)
{
  int i,b;
  outb (0x70,(0x80|addr));
  b = inb(0x71);
  i = (b & 15) + (b>>4)*10;
  if (i>99) printf("\ncmos_read: addr:%3d i:%3d(0x%02x) ",addr,i,b);
  return i;
}

static inline void cmos_write(int addr, int value)
{
  outb(0x70,0x80|addr);
  outb(0x71,((value/10) << 4) + value % 10);
}

time_t get_cmos_time()
{
  struct tm tm;
  int i;

  for (i = 0 ; i < 1000000 ; i++) /* must try at least 2.228 ms*/
    if (!(CMOS_READ(RTC_FREQ_SELECT) & RTC_UIP))
      break;
  
  do {
    tm.tm_sec  = cmos_read(RTC_SECONDS);
    tm.tm_min  = cmos_read(RTC_MINUTES);
    tm.tm_hour = cmos_read(RTC_HOURS);
/*  tm.tm_wday = cmos_read(RTC_DAY_OF_WEEK); */
    tm.tm_mday = cmos_read(RTC_DAY_OF_MONTH);
    tm.tm_mon  = cmos_read(RTC_MONTH);
    tm.tm_year = cmos_read(RTC_YEAR);
  } while (tm.tm_sec != cmos_read(RTC_SECONDS));

  tm.tm_mon  -= 1; /* DOS uses 0 base */
/*tm.tm_wday -= 3; / * DOS uses 3 - 9 for week days */
  tm.tm_isdst = dst; /* don't know whether it's daylight */

  return (mktime(&tm));
}

void set_cmos_time(struct tm *tm)
{
  unsigned char save_control, save_freq_select;

  save_control = CMOS_READ(RTC_CONTROL); /* tell the clock it's being set */
  CMOS_WRITE((save_control|RTC_SET), RTC_CONTROL);

  save_freq_select = CMOS_READ(RTC_FREQ_SELECT); /* stop and reset prescaler */
  CMOS_WRITE((save_freq_select|RTC_DIV_RESET2), RTC_FREQ_SELECT);
  cmos_write(RTC_SECONDS, tm->tm_sec);
  cmos_write(RTC_MINUTES, tm->tm_min);
  cmos_write(RTC_HOURS, tm->tm_hour);
  cmos_write(RTC_DAY_OF_WEEK, tm->tm_wday+3);
  cmos_write(RTC_DAY_OF_MONTH, tm->tm_mday);
  cmos_write(RTC_MONTH, tm->tm_mon+1);
  cmos_write(RTC_YEAR, tm->tm_year);

  CMOS_WRITE(save_freq_select, RTC_FREQ_SELECT);
  CMOS_WRITE(save_control, RTC_CONTROL);
}



/* ------------------------------------------------------------ */


void usr1_handler(int i)
{
  setUNIX = !setUNIX;
  if (setUNIX) setUNIXonce = !setUNIXonce;
  printf("%s setUNIX %s setUNIXonce\n"
	 ,setUNIX     ? "enable " : "disable"
	 ,setUNIXonce ? "enable " : "disable");
  fflush(stdout);
}

void usr2_handler(int i)
{
  setCMOS = !setCMOS;
  if (setCMOS) setCMOSonce = !setCMOSonce;
  printf("%s setCMOS %s setCMOSonce\n"
	 ,setCMOS     ? "enable " : "disable"
	 ,setCMOSonce ? "enable " : "disable");
  fflush(stdout);
}


/* ------------------------------------------------------------ */

void main(int argc,char *argv[])
{
  struct timeval t1,t2;
  long tt1=0,tt2=0 /* ,tt3=0,tt4=0 */ ;
  int i,p1=1,p2=1,p3=1;
  unsigned char c;
  char adj_msg[80];
  int start=3;

  int n=0;
  long b1=0,b2=0;
  int ok=0;
  int b=0,bb=0;
  char *dcf_line = DCF_TTY;
  int nsync=0, ticksync=0;
  struct tm tm;
  int opt;
  int nice_level = -10;

  int std,min, tag,mon,jahr,wtag;

  struct timeval t_dcf,adjust,last_adjust;
  time_t t_cmos;
  long diff_unix, diff_sum[3], diff_n[3];
  double diff_sum2[3];
  long tick[3][60];
  long sec_diff;
  struct timex tx;
  int refpos = 0;

  struct ppsclocknev ppsclocknev;
  struct timeval tttt;

  adj_msg[0] = '\0';
  std = min = tag = mon = jahr = wtag = 0;
  dst = last_dst = -1; dstchange = 0;
  for(i=0; i<=2; i++) {
    diff_sum[i] = diff_n[i]  = 0;
    /* if (stats) */ diff_sum2[i] = 0;
  }

  while ((opt=getopt(argc,argv,"jncuCUdst:D:R:N:lr:"))!=EOF) {
    switch (opt)
      {
      case 'j' : dcf_baud=B50;    break;
      case 'n' : dcf_baud=B38400; break;
      case 'c' : setCMOSonce = setCMOS = 1; break;
      case 'u' : setUNIXonce = setUNIX = 1; break;
      case 'C' : setCMOSonce = 1; break;
      case 'U' : setUNIXonce = 1; break;
      case 'd' : debug++; break;
      case 's' : stats++; break;
      case 't' : dcf_line = optarg; break;
      case 'R' : set_rts = atoi(optarg); break;
      case 'D' : set_dtr = atoi(optarg); break;
      case 'N' : nice_level = atoi(optarg); break;
      case 'l' : use_dcf_discipline = 0; break;
      case 'r' : refpos = atoi(optarg); break;
      default:   printf("usage: dcf  [-j] : use Junghans receiver (U2775B)\n"
			"            [-n] : use non-Junghans receiver\n"
			"            [-c] : set CMOS clock\n"
			"            [-C] : set CMOS clock only once\n"
			"            [-u] : set UNIX clock\n"
			"            [-U] : set UNIX clock only once\n"
			"            [-d] : debug mode\n"
			"            [-s] : stats output\n"
#if CIOGETNEV
			"            [-l] : don't use dcf tty line discipline\n"
			"            [-r n] : reference position [0,1,2]\n"
#endif
			"            [-R n] : set RTS 0=- [1=+]\n"
			"            [-D n] : set DTR 1=+ [0=-]\n"
			"            [-N n] : nice\n"
			"            [-t tty]\n");
	exit(1);
      }
  }

  if (!use_dcf_discipline) refpos = 0;
  nice(nice_level); 

  dcf_init(dcf_line);
  cmos_init();
  signal(SIGUSR1,usr1_handler);
  signal(SIGUSR2,usr2_handler);

  while (!read(dcf_dev,&c,1));
  gettimeofday(&t1,NULL);
  t_dcf = t1;
  t_dcf.tv_usec -= 190 * 1000;
  if (t_dcf.tv_usec<0) { 
    t_dcf.tv_sec--;
    t_dcf.tv_usec += 1000*1000;
  }
  if (use_dcf_discipline) {
    usleep(200*1000);
    if (ioctl(dcf_dev, CIOGETNEV, &ppsclocknev) < 0) {
      perror("dcf: ioctl(CIOGETNEV)");
      use_dcf_discipline = 0;
    }
    t1 = ppsclocknev.tv_raise;
    t1.tv_usec += 190 * 1000;
  }
  if (t_dcf.tv_usec > 500*1000) t_dcf.tv_sec++; 
  t_dcf.tv_sec++;
  if (use_dcf_discipline) t_dcf.tv_usec = refpos * 100*1000;
  else t_dcf.tv_usec  = 190 * 1000; /* 190ms fuer 9.5 bit @ 50 Baud */


  for (;;) {
    while (!read(dcf_dev,&c,1));
    if (use_dcf_discipline) {
      if (debug>1) gettimeofday(&tttt,NULL);
      usleep(200*1000);
      if (ioctl(dcf_dev, CIOGETNEV, &ppsclocknev) < 0) {
	perror("dcf: ioctl(CIOGETNEV)");
	use_dcf_discipline = 0;
      }
      t2 = ppsclocknev.tv_raise;
    }
    else {
      gettimeofday(&t2,NULL);
    }

    sec_diff = DCF_DIFF(t2,t1);
    if (sec_diff > 1500000L || n > 59) {
      if (n != 59 || sec_diff>2500000L || (p1|p2|p3)) {
	std = min = tag = mon = jahr = wtag =
	  ok = nsync = ticksync = 0;
	dst = -1; dstchange = 0;
	printf("\nnsync=0 n=%d %ld > %ld p=%d%d%d"
	       ,n,sec_diff,2500000L,p1,p2,p3);
	gettimeofday(&t_dcf,NULL);
	t_dcf.tv_sec = ((t_dcf.tv_sec + 30) / 60) * 60;
	if (use_dcf_discipline) {
	  t_dcf.tv_usec = refpos * 100*1000;
	}
	else {
	  t_dcf.tv_usec  = 190 * 1000; /* 190ms fuer 9.5 bit @ 50 Baud */
	  if (use_dcf_discipline) t_dcf.tv_usec += 200 * 1000; /* 200ms fuer usleep() */
	}
      }
      else {
	diff_unix = DCF_DIFF(t2,t_dcf);
	diff_unix = diff_sum[refpos] / diff_n[refpos];
	
	if (nsync>1 && tt2-tt1==60 /*3 && tt4-tt3==60 && tt3-tt2==60 && tt2-tt1==60*/) {
	  if (setUNIX || setUNIXonce) {
	    if ((start==1 || start==2) && nsync>2) {
	      int ds,dus;
	      struct timeval t,tt;
	      printf(",START%d",start);
	      start--;
	      ds  = -diff_unix/1000000;
	      dus = -diff_unix - ds*1000000;
	      usleep(10*1000);
	      gettimeofday(&t,NULL);
	      tt=t;
	      t.tv_sec  += ds;
	      t.tv_usec += dus;
	      if (t.tv_usec<0)            { t.tv_usec += 1000000; t.tv_sec--; }
	      else if (t.tv_usec>1000000) { t.tv_usec -= 1000000; t.tv_sec++; }
	      settimeofday(&t,NULL);
	      t2.tv_sec  += ds;
	      t2.tv_usec += dus;
	      if (t2.tv_usec<0)            { t2.tv_usec += 1000000; t2.tv_sec--; }
	      else if (t2.tv_usec>1000000) { t2.tv_usec -= 1000000; t2.tv_sec++; }
	      ppsclocknev.tv_fall.tv_sec  += ds;
	      ppsclocknev.tv_fall.tv_usec += dus;
	      if (ppsclocknev.tv_fall.tv_usec<0)            { ppsclocknev.tv_fall.tv_usec += 1000000; ppsclocknev.tv_fall.tv_sec--; }
	      else if (ppsclocknev.tv_fall.tv_usec>1000000) { ppsclocknev.tv_fall.tv_usec -= 1000000; ppsclocknev.tv_fall.tv_sec++; }
	    }
	    else if (start) {
	      printf(",START%d",start);
	      start = 2;
	      t_dcf.tv_usec  = 190 * 1000; /* 190ms fuer 9.5 bit @ 50 Baud */
	      if (use_dcf_discipline) t_dcf.tv_usec += 200 * 1000; /* 200ms fuer usleep() */
	      settimeofday(&t_dcf,NULL);
	      t_dcf.tv_usec = refpos * 100*1000;
	      ppsclocknev.tv_fall = t2 = t_dcf;
	      t2.tv_usec = use_dcf_discipline ? 0 : 190 * 1000; /* 190ms fuer 9.5 bit @ 50 Baud */
	      ppsclocknev.tv_fall.tv_usec = 100*1000;
	    }
	    else if (diff_unix>-1000000/10 && diff_unix<1000000/10) {  /* -100ms ... 100ms */
	       if ((double)(diff_unix)*(double)(diff_unix)
		   > diff_sum2[refpos]/diff_n[refpos]/2) {
		  /* diff_unix > std. deviation */
		  adjust.tv_sec  = 0;
		  adjust.tv_usec = -diff_unix;
		  adjtime(&adjust,&last_adjust);
	       }
	    }
	    else {
	      t_dcf.tv_usec  = 190 * 1000; /* 190ms fuer 9.5 bit @ 50 Baud */
	      if (use_dcf_discipline) t_dcf.tv_usec += 200 * 1000; /* 200ms fuer usleep() */
	      settimeofday(&t_dcf,NULL);
	      t_dcf.tv_usec = refpos * 100*1000;
	    }
	    printf(",setUNIX");
	    if (start==0) setUNIXonce = 0;
	  }
	  tx.modes = 0;
	  adjtimex(&tx);
	  sprintf(adj_msg,"f=%ld ",tx.freq);
	  
	  t_cmos = get_cmos_time();

	  if (setCMOS || setCMOSonce || last_dst!=dst) {
	    set_cmos_time(&tm);
	    printf(",setCMOS");
	    last_dst = dst;
	    setCMOSonce = 0;
	  }
	}
	else t_cmos = get_cmos_time();
	printf(" udiff= %9.6f cdiff= %2ld", diff_unix*1e-6, t_cmos-t_dcf.tv_sec);
	ok = 1;
      }
      printf(" ok=%d %08lx %08lx %s%9d\r",ok,b1,b2,adj_msg,t_dcf.tv_sec);
      adj_msg[0] = '\0';
      if (!(setCMOS||setUNIX) || debug) putchar('\n');
      fflush(stdout);
      n = b1 = b2 = 0;
      bb = 1;
      for(i=0; i<=2; i++) {
	diff_sum[i] = diff_n[i]  = 0;
	/* if (stats) */ diff_sum2[i] = 0;
      }
if (!start) {
      t_dcf.tv_sec = t2.tv_sec;
      if (t2.tv_usec>1000000/2) t_dcf.tv_sec++;
}
      if (use_dcf_discipline) {
	t_dcf.tv_usec = refpos * 100*1000;
      }
      else {
	t_dcf.tv_usec  = 190 * 1000; /* 190ms fuer 9.5 bit @ 50 Baud */
	if (use_dcf_discipline) t_dcf.tv_usec += 200 * 1000; /* 200ms fuer usleep() */
      }

      fflush(stdout);
    }

    if (use_dcf_discipline) {
      i = (c<c2b) + 1;
      diff_unix = DCF_DIFF(ppsclocknev.tv_fall,t_dcf)-(n-0)*1000*1000;
      tick[i][diff_n[i]++] = diff_unix;
    }

    diff_unix = DCF_DIFF(t2,t_dcf)-(n-0)*1000000;
    tick[0][diff_n[0]++] = diff_unix;

    if (stats>1) {
      if (use_dcf_discipline) 
	printf("%2d   %6ld %6d   "
	       ,n,diff_unix,DCF_DIFF(ppsclocknev.tv_fall,t_dcf)-(n-0)*1000000
	       );
      else
	printf("%2d   %6ld   "
	       ,n,diff_unix
	       );
      if (!(debug>1) ) {
	if (stats<2) printf("\n");
	fflush(stdout);
      }
    }

    if (debug>1) {
      printf("%02d:%02d:%02d %02d.%02d.%02d %s "
	     ,std,min, n ,tag,mon,jahr,wtag_txt[wtag]);
      if (use_dcf_discipline) 
	printf("%02x->%d %10d.%06d %10d.%06d %10d.%06d %10d.%06d\n"
	       ,c,c<c2b /* , sec_diff*1e-6, sec_diff*1e-6-1 */ 
	       ,PF(ppsclocknev.tv_raise) /* , ppsclocknev.serial_raise */
	       ,PF(ppsclocknev.tv_fall)  /* , ppsclocknev.serial_fall  */
	       ,PF(ppsclocknev.tv_data)  /* , ppsclocknev.serial_data  */
	       ,PF(tttt)
	   /*    ,DCF_DIFF(ppsclocknev.tv_fall,ppsclocknev.tv_raise)*1e-6 
	       ,DCF_DIFF(tttt,ppsclocknev.tv_raise)*1e-6 */
	       );
      else
	printf("%02x->%d %9.6f %10.6f\n"
	       ,c,c<c2b, sec_diff*1e-6, sec_diff*1e-6-1
	       );
      fflush(stdout);
    }

    if (c<c2b) {
      if (n<29) b1 |= bb;
      else      b2 |= bb;
    }
    if (++n == 29) bb = 1;
    else bb <<= 1;
    
    if (n==59) {
      nsync++;
      for (p1=0,i=21,b=1<<i; i<=28; i++, b<<=1) if (b1&b) p1 ^= 1;
      for (p2=0,i= 0,b=1<<i; i<= 6; i++, b<<=1) if (b2&b) p2 ^= 1;
      for (p3=0,i= 7,b=1<<i; i<=29; i++, b<<=1) if (b2&b) p3 ^= 1;

      dstchange = ((b1>>16) & 0x01);
      switch ((b1>>17) & 0x03) 	{
        case 2:  dst =  0; break;
        case 1:  dst =  1; break;
        default: dst = -1;
      }

      min = ((b1>>21) & 0x0f) + ((b1>>25)&0x07) * 10;
      std = ((b2>> 0) & 0x0f) + ((b2>> 4)&0x03) * 10;
      tag = ((b2>> 7) & 0x0f) + ((b2>>11)&0x03) * 10;
      wtag= ((b2>>13) & 0x07);
      mon = ((b2>>16) & 0x0f) + ((b2>>20)&0x01) * 10;
      jahr= ((b2>>21) & 0x0f) + ((b2>>25)&0x0f) * 10;

      tt1=tt2; /* tt2=tt3; tt3=tt4; */
      tm.tm_sec   = 0;
      tm.tm_min   = min;
      tm.tm_hour  = std;
      tm.tm_wday  = 0;
      tm.tm_mday  = tag;
      tm.tm_mon   = mon-1;
      tm.tm_year  = jahr;
      tm.tm_isdst = dst;
      tt2 = t_dcf.tv_sec = mktime(&tm);

      if (use_dcf_discipline) {
	t_dcf.tv_usec = refpos * 100*1000;
      }
      else {
	t_dcf.tv_usec  = 190 * 1000; /* 190ms fuer 9.5 bit @ 50 Baud */
	if (use_dcf_discipline) t_dcf.tv_usec += 200 * 1000; /* 200ms fuer usleep() */
      }

      for (i=0; i<=2; i++) {
	int j,n, n0,min,max,diff_unix;
	n = diff_n[i];
	n0 = (n+5)/10; 
	if (n0<2) n0 = 2;
	min = max = 0;
	for (;n0--;) {
	  for (j=1; j<n; j++) {
	    if (tick[i][min] > tick[i][j]) min = j;
	    else if (tick[i][max] < tick[i][j]) max = j;
	  }
/*	  printf("minmax %d   %2d %8ld   %2d %8ld\n",i,min,tick[i][min],max,tick[i][max]); */
	  if (n) tick[i][min] = tick[i][--n];
	  if (n) if (max!=min) tick[i][max] = tick[i][--n];
	}
	diff_n[i] = n;
	diff_sum[i] = 0;
	/* if (stats) */ diff_sum2[i] = 0;
	for (j=0; j<n; j++) {
	  diff_unix = tick[i][j];
	  diff_sum[i]  += diff_unix;
	  /* if (stats) */ { 
	    diff_sum2[i] += (double)(diff_unix) * (double)(diff_unix);
	  }
	}
      }
      if (stats) {
	if (use_dcf_discipline) 
	  printf("%2d   %6ld,%-6ld  %6ld,%-6ld  %6ld,%-6ld\n"
		 ,n
		 ,diff_sum[0]/diff_n[0],(long)(sqrt(diff_sum2[0]/diff_n[0] - (double)(diff_sum[0])/diff_n[0] * (double)(diff_sum[0])/diff_n[0]))
		 ,diff_sum[1]/(diff_n[1]>0?diff_n[1]:1),(long)(sqrt(diff_sum2[1]/(diff_n[1]>0?diff_n[1]:1) - (double)(diff_sum[1])/(diff_n[1]>0?diff_n[1]:1) * (double)(diff_sum[1])/(diff_n[1]>0?diff_n[1]:1)))
		 ,diff_sum[2]/(diff_n[2]>0?diff_n[2]:1),(long)(sqrt(diff_sum2[2]/(diff_n[2]>0?diff_n[2]:1) - (double)(diff_sum[2])/(diff_n[2]>0?diff_n[2]:1) * (double)(diff_sum[2])/(diff_n[2]>0?diff_n[2]:1)))
		 );
	else
	  printf("%2d   %6ld,%-6ld\n"
		 ,n
		 ,diff_sum[0]/diff_n[0],(long)(sqrt(diff_sum2[0]/diff_n[0] - (double)(diff_sum[0])/diff_n[0] * (double)(diff_sum[0])/diff_n[0]))
		 );
      }
      printf("%02d:%02d:%02d %02d.%02d.%02d %s "
	     ,std,min, 0 ,tag,mon,jahr,wtag_txt[wtag]);
      printf("nsync=%-3d p=%d%d%d"
	     ,nsync,p1,p2,p3);
      fflush(stdout);

    }
    t1 = t2;
  }
}
